/*
 * TagTreeControl.cpp
 *
 *  Created on: 23.10.2009
 *      Author: stefan.detter
 */

#include "TagTreeControl.h"

#include <QMessageBox>

#include "TagTreeModel.h"
#include "TagTreeProxy.h"
#include "TagTreeDelegate.h"
#include "TagTreeContextMenu.h"

#include "../data/TreeItem.h"
#include "../data/RootItem.h"
#include "../data/ReaderEntry.h"
#include "../data/TagEntry.h"

#include "../../application/Application.h"

#include <def.h>


TagTreeControl::TagTreeControl(Application* application, TreeView* treeView, QPushButton* clearButton, QPushButton* backButton,
		QLabel* selectedReaderLabel, QLabel* selectedReaderPixmapLabel, QWidget* window, QObject* parent)
	: QObject(parent)
	, m_application(application)
	, m_treeView(treeView)
	, m_clearButton(clearButton)
	, m_backButton(backButton)
	, m_selectedReaderLabel(selectedReaderLabel)
	, m_selectedReaderPixmapLabel(selectedReaderPixmapLabel)
	, m_window(window)
{
	m_tagTreeModel = new TagTreeModel(this);
	m_tagTreeProxy = new TagTreeProxy(this);
	m_tagTreeDelegate = new TagTreeDelegate(m_tagTreeProxy, this);
	m_tagTreeContextMenu = new TagTreeContextMenu(m_treeView, m_tagTreeModel, m_tagTreeProxy, m_window, this);

	m_tagTreeProxy->setSourceModel(m_tagTreeModel);
	m_tagTreeProxy->setDynamicSortFilter(true);

	m_treeView->setItemDelegate(m_tagTreeDelegate);
	m_treeView->setModel(m_tagTreeProxy);
	m_treeView->setSortingEnabled(true);
	m_treeView->sortByColumn(0, Qt::AscendingOrder);

	m_treeView->setColumnPercentWidth(0, 50);
	m_treeView->setColumnPercentWidth(1, 10);
	m_treeView->setColumnPercentWidth(2, 10);
	m_treeView->setColumnPercentWidth(3, 10);
	m_treeView->setColumnPercentWidth(4, 10);
	m_treeView->setColumnPercentWidth(5, 10);

	connect(m_application, 			SIGNAL(readerAttached(Reader*)),
			m_tagTreeModel, 		  SLOT(readerAttached(Reader*)) );

	connect(m_tagTreeModel, 		SIGNAL(expand(const QModelIndex&)),
			m_tagTreeProxy, 		  SLOT(expandSlot(const QModelIndex&)) );
	connect(m_tagTreeProxy, 		SIGNAL(expand(const QModelIndex&)),
			m_treeView, 			  SLOT(expand(const QModelIndex&)) );

	connect(m_tagTreeModel, 		SIGNAL(setCurrentIndex(const QModelIndex&)),
			m_tagTreeProxy, 		  SLOT(setCurrentIndexSlot(const QModelIndex&)) );
	connect(m_tagTreeProxy, 		SIGNAL(setCurrentIndex(const QModelIndex&)),
			m_treeView, 			  SLOT(setCurrentIndex(const QModelIndex&)) );

	connect(m_tagTreeModel, 		SIGNAL(setRootIndex(const QModelIndex&)),
			m_tagTreeProxy, 		  SLOT(setRootIndexSlot(const QModelIndex&)) );
	connect(m_tagTreeProxy, 		SIGNAL(setRootIndex(const QModelIndex&)),
			m_treeView, 			  SLOT(setRootIndex(const QModelIndex&)) );

	connect(m_treeView, 			SIGNAL(clickedIntoFreeSpace()),
			m_tagTreeModel, 		  SLOT(clickedIntoFreeSpace()) );

	connect(m_treeView, 			SIGNAL(clicked(const QModelIndex&)),
			m_tagTreeProxy, 		  SLOT(clickedSlot(const QModelIndex&)) );
	connect(m_tagTreeProxy, 		SIGNAL(clicked(const QModelIndex&)),
			m_tagTreeModel, 		  SLOT(clicked(const QModelIndex&)) );

	connect(m_treeView, 			SIGNAL(doubleClicked(const QModelIndex&)),
			m_tagTreeProxy, 		  SLOT(doubleClickedSlot(const QModelIndex&)) );
	connect(m_tagTreeProxy, 		SIGNAL(doubleClicked(const QModelIndex&)),
			m_tagTreeModel, 		  SLOT(doubleClicked(const QModelIndex&)) );

	connect(m_tagTreeModel, 		SIGNAL(itemClicked(const TreeItem*)),
			this, 					SIGNAL(itemClicked(const TreeItem*)) );
	connect(m_tagTreeModel, 		SIGNAL(itemSetRoot(const TreeItem*)),
			this, 					  SLOT(itemSetRoot(const TreeItem*)) );
	connect(m_tagTreeModel, 		SIGNAL(restoredRootIndex()),
			this, 					  SLOT(restoreReaderSelection()) );

	m_backButton->parentWidget()->setVisible(false);

	connect(m_backButton, 			SIGNAL(clicked()),
			m_tagTreeModel, 		  SLOT(restoreRootIndex()));
	connect(m_clearButton, 			SIGNAL(clicked()),
			m_tagTreeModel, 		  SLOT(clearTags()));

	connect(m_treeView, 			SIGNAL(customContextMenuRequested ( const QPoint& )),
			m_tagTreeContextMenu, 	  SLOT(showPopup(const QPoint &)));
	connect(m_tagTreeContextMenu, 	SIGNAL(appendGlobalJob(JobType, const QList<QVariant>&)),
			this,  					SIGNAL(appendGlobalJob(JobType, const QList<QVariant>&)));
	connect(m_tagTreeContextMenu, 	SIGNAL(addReaderDialogRequested()),
			this, 					SIGNAL(addReaderDialogRequested()));

}

TagTreeControl::~TagTreeControl()
{
}

TreeItem* TagTreeControl::selectedItem()
{
	return m_tagTreeModel->selectedItem();
}

RootItem* TagTreeControl::rootItem()
{
	return m_tagTreeModel->rootItem();
}

void TagTreeControl::restoreReaderSelection()
{
	m_backButton->parentWidget()->setVisible(false);
	emit readerUnselected();
}

void TagTreeControl::itemSetRoot ( const TreeItem* t )
{
	if(t->itemType() != TreeItem::READER)
		return;

	const ReaderEntry* r = dynamic_cast<const ReaderEntry*>(t);
	if(r == 0)
		return;

	m_selectedReaderLabel->setText ( r->data(0, Qt::DisplayRole).toString() );
	if( t->data(0, Qt::DecorationRole).convert(QVariant::Pixmap) )
		m_selectedReaderPixmapLabel->setPixmap (  r->data(0, Qt::DecorationRole).value<QPixmap>() );
	m_backButton->parentWidget()->setVisible(true);
	emit readerSelected(r->reader());
}


void TagTreeControl::removeOfflineReader()
{
	m_tagTreeModel->rootItem()->removeOfflineReader();
}
